#!/usr/bin/perl

use strict 'vars';

use Archive::Zip qw( :ERROR_CODES );
use DBI;
use IO::Handle;
use File::Copy;
use File::Find;
use File::Path;
use FindBin;
use LWP::UserAgent;
use POSIX qw/strftime/;

our ($Dir, $Name, $Virtual, %In);
our ($Language, %LangMessages);
our ($HeaderPrinted, $FooterPrinted);
our (@Profiles, %Cfg, @FormFields, @Actions);
our ($DB, %Tmp, $Log);

STDOUT->autoflush(1);
$| = 1;

#
# special DIE processing

BEGIN
{
	$SIG{__DIE__} = sub
	{
		my $msg = shift;
		
		warn($msg);

		print "Pragma: no-cache\nCache-control: no-cache\nContent-type: text/plain\n\n";
		print "DIED: $msg\n";

		my $i = 0;
		while (my @trace = caller($i++))
		{
			printf("\t%s:%s %s\n", $trace[1], @trace[2,3]);
		} 

		exit(1);
	};
}

#
# subs

sub init
{
	$Dir = $FindBin::Bin;
	($Name = $FindBin::Script) =~ s/\.\w+$//;
	($Virtual = $ENV{SCRIPT_NAME}) =~ s![^/]+$!!;
	
	my $in = {};
	my $postbuffer = undef;

	my @pairs;
	if ($ENV{CONTENT_TYPE} =~ m#^multipart/form-data#)
	{
		return _parse_multipart();
	}
	elsif (defined $ENV{CONTENT_LENGTH} && $ENV{CONTENT_LENGTH} > 0)
	{
		read(STDIN, $postbuffer, $ENV{CONTENT_LENGTH});
		@pairs = split(/&/, $postbuffer);
	}
	elsif ($ENV{QUERY_STRING} ne '')
	{
		@pairs = split(/&/, $ENV{QUERY_STRING});
	}
	else
	{
		return $in;
	}

	my $name;
	for (@pairs)
	{
		($name, $_) = split '=';
		tr/+/ /;
		s/%([a-fA-F0-9][a-fA-F0-9])/pack("C", hex($1))/eg;
		$In{$name} = $In{$name} ? "$in->{$name},$_" : $_;
	}

	return;
}

sub printHeader
{
	my $title = shift;
	
	return if $HeaderPrinted;
	
	print <<END_HTML;
Pragma: no-cache\nCache-control: no-cache\nContent-type: text/html

<html>
<head>
<title>$title</title>
</head>
<body>
	
<center>	
<table style="margin: 5px; background-color: rgb(0, 76, 153); font-family: normal Tahoma,Verdana,sans-serif; font-style: normal; font-variant: normal; font-weight: normal; font-size: 11pt; line-height: normal; font-size-adjust: none; font-stretch: normal;" border="0" cellpadding="5" cellspacing="1" width=30%>
<tbody>
<tr><td align="center"><span style="color: white; font-weight: bold;">$title<span></td></tr>
<tr>
	<td bgcolor="white">
END_HTML

	$HeaderPrinted = 1;
	
	return;
}

sub printFooter
{
	return if $FooterPrinted;

	print <<END_HTML;
	
	</td>
</tr>
</tbody></table>
</center>	
</body>
</html>
END_HTML

	$FooterPrinted = 1;
	
	return;
}

sub outputError { output('error', @_) }
sub outputMessage { output('', @_) }

sub output
{
	my $type = shift;
	my $message = shift;
	my $print_form = shift;
	
	printHeader($Cfg{Title});
	
	my $styleAdd = "color:red;"
		if $type eq 'error';
	print qq(<div style="text-align:center;$styleAdd">$message</div>);
	
	# form: 1 - main form; 2 - profile selection form
	if ($print_form == 1)
	{
		printMainForm();
	}
	elsif ($print_form == 2)
	{
		printProfileForm();
	}
	
	printFooter();
	
	exit;
}

sub loadProfiles
{
	my $mainZipFile = "$Dir/$Name.zip";
	my $errmsg = "Main zip file $mainZipFile not found or damaged";

	outputError("$errmsg (1)")
		unless -e $mainZipFile;

	my $zip = Archive::Zip->new();

	outputError("$errmsg (2)")
		unless $zip->read($mainZipFile) == AZ_OK;

	for my $member (map { $_->fileName } $zip->membersMatching(qr/^installer\.profile\./))
	{
		my $tmpfile = "$Dir/$Name.profile";
		outputError("$errmsg (3)")
			unless $zip->extractMember($member, $tmpfile) == AZ_OK;
		open F, $tmpfile or outputError("$errmsg (4)");
		while (my $line = <F>)
		{
			$line =~ s/[\r\n]//g;
			
			if ($line =~ /^LangOptionFile\=(.+)/)
			{
				scanLangData($zip, *F, $1);
			}
			elsif ($line =~ /^ProfileProduct\=(.+)/ && ! $Cfg{Title})
			{
				$Cfg{Title} = $1;
				langifyString(\($Cfg{Title}))
					if $Language;
			}
			elsif ($line =~ /^ProfileOption\=(.+)/)
			{
				my $profile_title = $1;
				langifyString(\$profile_title)
					if $Language;
				push @Profiles, [ $member, $profile_title ];
				last;
			}
		}
		close F;
		unlink $tmpfile;
	}
	
	outputError("$errmsg (5)")
		unless @Profiles;

	return;
}

sub loadConfig
{
	my $profile = shift;

	my $mainZipFile = "$Dir/$Name.zip";
	my $errmsg = "Error loading profile $profile from $mainZipFile";

	outputError("$errmsg (1)")
		unless -e $mainZipFile;

	my $zip = Archive::Zip->new();

	outputError("$errmsg (2)")
		unless $zip->read($mainZipFile) == AZ_OK;

	my $tmpfile = "$Dir/$Name.profile";

	outputError("$errmsg (3)")
		unless $zip->extractMember($profile, $tmpfile) == AZ_OK;

	my $currAction;

	open F, $tmpfile or outputError("$errmsg (4)");
	local $_;
	while ($_ = <F>)
	{
		s/^\s+//g;
		s/[\r\n\s]+$//g;
		next
			unless length;
		next
			if /^#/;
			
		my ($k, $v) = split /\s*\=\s*/, $_, 2;
			
		# langify value
		langifyString(\$v)
			if $Language;

		if ($k eq 'LangOptionFile')
		{
			scanLangData($zip, *F, $v);
		}
		elsif ($k eq 'Action')
		{
			$currAction = {
				name => $v,
				param => [],
			};
			push @Actions, $currAction;
		}
		elsif ($k eq 'Param')
		{
			die unless $currAction;
			push @{ $currAction->{param} }, $v;
		}
		else
		{
			$Cfg{$k} = $v;
			
			if (/^Form\.(.+?)\.title/)
			{
				push @FormFields, $1;
			}
		}
	}

	close F;
	
	unlink $tmpfile;
	
	return;
}

sub scanLangData
{
	my ($main_zip, $profile_fh, $lang_option_file) = @_;

	my $tmpfile = "$Dir/$Name.profile.lang";
	my $errmsg = "Error loading lang option file $tmpfile from the main zip file";

	outputError("$errmsg (1)")
		unless $main_zip->extractMember($lang_option_file, $tmpfile) == AZ_OK;

	open FL, $tmpfile or outputError("$errmsg (2)");
	read FL, $Language, -s FL;
	close FL;
	$Language =~ s/[\s\r\n]//g;
	
	unlink $tmpfile;

	while ($_ = <$profile_fh>)
	{
		s/^\s+//g;
		s/[\r\n\s]+$//g;
		next
			unless length;
		next
			if /^#/;
			
		my ($k, $v) = split /\s*\=\s*/, $_, 2;
	
		if ($k =~ /^LangMessage\.(\w+)\.(\w+)/)
		{
			$LangMessages{$1}{$2} = $v;
		}
		else
		{
			last;
		}
	}

	return;
}

sub langifyString
{
	my $str_ref = shift;
	# $1 - code, $2 - lang, $3 - text
	$$str_ref =~ s|\[\:\:L\s+(\w+)\s+\#\#(\w+)\s+(.*?)\s*\:\:\]|$2 eq $Language ? $3 : $LangMessages{$Language}{$1}|ge;
	return;
}

sub printProfileForm
{
	print qq(<form><table align="center">);
	print qq(<tr><td>&nbsp;</td></tr>);
	
	for (@Profiles)
	{
		print qq(<tr><td><nobr><input type="radio" name="installer_profile" value="$_->[0]">&nbsp;$_->[1]</nobr></td></tr>);
	}

	print qq(<tr><td>&nbsp;</td></tr>);
	print qq(<tr><td align="center"><input type="submit" value="Proceed"></td></tr>);
	print qq(</table></form>);
}

sub printMainForm
{
	print qq(<form><input type="hidden" name="installer_profile" value="$In{installer_profile}"><table align="center">);
	print qq(<tr><td colspan="2">&nbsp;</td></tr>);
	for my $f (@FormFields)
	{
		my $title = $Cfg{"Form.$f.title"};
		$title =~ s/\s+/&nbsp;/g;
		my $value = exists($In{$f}) ? $In{$f} : $Cfg{"Form.$f.default"} ? $Cfg{"Form.$f.default"} : undef;
		$value =~ s/\"/&quot;/g;
		print qq(<tr><td>$title</td><td><input type="text" size="50" name="$f" value="$value"></td></tr>);
	}
	print qq(<tr><td colspan="2" align="center"><input type="submit" name="do_it" value="$Cfg{TitleSubmit}"></td></tr>);
	print qq(</table></form>);
}

sub addlog
{
	my ($action, $msg) = @_;
	return
		unless $Log;
	print $Log sprintf("[%s] %-12s %s\n", scalar(localtime), $action, $msg);
	return;
}

#
# actions

sub actionFormRequired
{
	my ($fieldlist, $errmsg) = @_;
	
	# check required fields
	my @missed = grep { ! length($In{$_}) } split(/\s*,\s*/, $fieldlist);
	outputError($errmsg . join(', ', map { $Cfg{"Form.$_.title"} } @missed), 1)
		if @missed;
	
	return;
}

sub actionFormValidate
{
	my ($field, $condition, $errmsg) = @_;

	# validates form field
	for ($In{$field})
	{
		my $ok = eval $condition;
		die $@ if $@;
		outputError($errmsg, 1)
			unless $ok;
	}
	
	return;
}

sub actionEvalCheck
{
	my ($eval, $errmsg) = @_;

	my $check = eval $eval;
	
	die $@ if $@;
	
	if ($check)
	{
		outputError($errmsg, 1);
	}
	
	return;
}

sub actionStartLog
{
	my ($filetmpl, $errmsg) = @_;

	my $file = strftime($filetmpl, localtime(time));
	open $Log, ">$file"
		or outputError(sprintf($errmsg, $file, $!), 1);
	
	$Log->autoflush(1);
	
	addlog('StartLog', "OK profile $In{installer_profile}");
	
	return;
}

sub actionEndLog
{
	addlog('EndLog', 'OK');

	close $Log
		if $Log;
	
	return;
}

sub actionMySQLCheck
{
	my ($dbsource, $dbuser, $dbpass, $errmsg) = @_;

	evalParam() for $dbsource, $dbuser, $dbpass, $errmsg;

	# check DB connection
	$DB = DBI->connect($dbsource, $dbuser, $dbpass)
		or outputError($errmsg, 1);
	
	addlog('MySQLCheck', 'OK');
	
	return;
}

sub actionMySQLImport
{
	my ($sqlFile, $skipCrit) = @_;

	evalParam() for $sqlFile;

	my $sqlFile = "$Dir/$sqlFile";

	# check skip criterion
	if ($skipCrit)
	{
		my $sth = $DB->prepare($skipCrit)
			or return addlog('MySQLImport', "skipping MySQLImport due to skipCrit compile error: [$skipCrit]");
		$sth->execute
			or return addlog('MySQLImport', "skipping MySQLImport due to skipCrit execute error: [$skipCrit]");
		my $row = $sth->fetchrow_arrayref;
		if ($row && $row->[0])
		{
			addlog('MySQLImport', "skipCrit positive: [$row->[0]]");
			return;
		}
		else
		{
			addlog('MySQLImport', "skipCrit negative");
		}
	}
	
	# read SQL file
	my $fh;
	open $fh, $sqlFile or die "failed to read $sqlFile: $!";
	my $sql;
	read $fh, $sql, -s $fh;
	close $fh;
	
	# split sql
	my $sqls = splitSqlFile($sql);
	
	# do sql queries
	for my $sql (@$sqls)
	{
		my $sth = $DB->prepare($sql)
			or do { addlog('MySQLImport', sprintf("SQL Error: (%s) %s in query [%s]\n", $DB->err, $DB->errstr, $sql)); next };
		$sth->execute
			or do { addlog('MySQLImport', sprintf("SQL Error: (%s) %s in query [%s]\n", $DB->err, $DB->errstr, $sql)); next };
		addlog('MySQLImport', "SQL OK: $sql");
	}
	
	addlog('MySQLImport', 'OK');
	
	return;
}

sub actionMySQLUpdate
{
	my ($v1, $v2, $sqlUpdDir) = @_;

	evalParam() for $v1, $v2, $sqlUpdDir;
	
	return
		if $v1 ge $v2;
	
	my $dh;
	opendir $dh, "$Dir/$sqlUpdDir" or die "Failed to open SQL updates directory [$sqlUpdDir]: $!";
	my @files = readdir $dh;
	closedir $dh;
	
	for my $f (grep { $_ gt "$v1.sql" && $_ le "$v2.sql" } sort { $a cmp $b } grep { /\.sql$/ } @files)
	{
		addlog('MySQLUpdate', "Import SQL file $sqlUpdDir/$f");
		actionMySQLImport("$sqlUpdDir/$f");
		addlog('MySQLUpdate', "OK SQL file $sqlUpdDir/$f");
	}
	
	for my $f (grep { $_ gt "$v1.pl" && $_ le "$v2.pl" } sort { $a cmp $b } grep { /\.pl$/ } @files)
	{
		addlog('MySQLUpdate', "Import PL file $sqlUpdDir/$f");
		my $sub = require "$sqlUpdDir/$f";
		&$sub($DB);
		addlog('MySQLUpdate', "OK PL file $sqlUpdDir/$f");
	}
	
	addlog('MySQLUpdate', 'OK');
	
	return;
}	

sub actionFileExists
{
	my ($file, $msgFalse, $msgTrue) = @_;

	evalParam() for $file, $msgFalse, $msgTrue;

	outputError($msgFalse, 1)
		if $msgFalse && ! -e $file;

	outputError($msgTrue, 1)
		if $msgTrue && -e $file;
	
	addlog('FileExists', "OK $file");
	
	return;
}

sub actionFileMkdir
{
	my ($dir, $errmsg) = @_;

	evalParam() for $dir, $errmsg;
	
	# create a directory
	-d "$FindBin::Bin/$dir" or mkdir("$FindBin::Bin/$dir") or outputError($errmsg, 1);
	
	addlog('FileMkdir', "OK $dir");
	
	return;
}

sub actionFileChmod
{
	my ($file, $mode) = @_;

	evalParam() for $file;
	
	if ($file =~ /\*/)
	{
		for (glob($file))
		{
			chmod oct($mode), $_;
		}
	}
	else
	{
		chmod oct($mode), $file;
	}
	
	addlog('FileChmod', "OK $file $mode");
	
	return;
}

sub actionFileCopy
{
	my ($f1, $f2, $errmsg) = @_;

	evalParam() for $f1, $f2, $errmsg;
	
	# copy file
	if ($f1 ne $f2)
	{
		copy("$FindBin::Bin/$f1", "$FindBin::Bin/$f2")
			or outputError($errmsg, 1);
		addlog('FileCopy', "OK copy $f1 to $f2");
	}
	else
	{
		addlog('FileCopy', "OK $f1 was not copied into itself");
	}
	
	return;
}

sub actionFileRename
{
	my ($f1, $f2, $errmsg) = @_;

	evalParam() for $f1, $f2, $errmsg;
	
	# rename file
	if ($f1 ne $f2)
	{
		move("$FindBin::Bin/$f1", "$FindBin::Bin/$f2")
			or outputError($errmsg, 1);
		addlog('FileCopy', "OK move $f1 to $f2");
	}
	else
	{
		addlog('FileCopy', "OK $f1 was not moved into itself");
	}
	
	return;
}

sub actionFileRemove
{
	my (@files) = @_;

	evalParam() for @files;
	
	unlink(map { "$Dir/$_"} @files);
	
	addlog('FileRemove', "OK @files");
	
	return;
}

sub actionFileRmtree
{
	my ($dir) = @_;

	evalParam() for $dir;
	
	rmtree($dir);
	
	addlog('FileRmtree', "OK $dir");
	
	return;
}

sub actionFilePasstree
{
	my ($dir, $regex, $eval) = @_;

	evalParam() for $dir;
	
	my $r = qr/$regex/i;

	File::Find::finddepth(
		sub
		{
			if ($_ =~ $r)
			{
				eval $eval;
				warn "ERROR during eval [$eval]: $@\n"
					if $@;
			}
		},
		$dir
	);
	
	addlog('FilePasstree', "OK $dir [$regex] [$eval]");
	
	return;
}

sub actionArchUnzip
{
	my ($file, $destDir, $errmsg) = @_;

	evalParam() for $file, $destDir, $errmsg;

	my $file = "$Dir/$file";
	my $dir = "$Dir/$destDir/";

	outputError($errmsg)
		unless -e $file;

	my $zip = Archive::Zip->new();

	outputError($errmsg)
		unless $zip->read($file) == AZ_OK;

	outputError($errmsg)
		unless $zip->extractTree('', $dir ) == AZ_OK;
	
	addlog('ArchUnzip', "OK $file $dir");

	return;
}

sub actionArchUnzipFile
{
	my ($zipFile, $inFile, $outFile, $errmsg) = @_;

	evalParam() for $zipFile, $inFile, $outFile, $errmsg;
	
	my $zipFile = "$Dir/$zipFile";
	my $outFile = "$Dir/$outFile";

	outputError($errmsg)
		unless -e $zipFile;

	my $zip = Archive::Zip->new();

	outputError($errmsg)
		unless $zip->read($zipFile) == AZ_OK;

	outputError($errmsg)
		unless $zip->extractMember($inFile, $outFile) == AZ_OK;
	
	addlog('ArchUnzipFile', "OK $zipFile $inFile $outFile");
	
	return;
}

sub actionDataModify
{
	my ($file, @modify) = @_;

	evalParam() for $file;
	
	# read file
	my $fh;
	open $fh, $file or die "failed to read $file: $!";
	my $data;
	read $fh, $data, -s $fh;
	close $fh;
	
	# modify
	while (my $regex = shift(@modify))
	{
		my $subst = shift(@modify);

		evalParam() for $subst;

		$data =~ s!$regex!$subst!gsi;

		addlog('DataModify', "OK $file [$regex] [$subst]")
	}
	
	# write file
	open $fh, ">$file" or die "failed to write $file: $!";
	binmode $fh;
	print $fh $data;
	close $fh;
	
	return;
}

sub actionDataScan
{
	my ($file, @scan) = @_;

	evalParam() for $file;
	
	# read file
	my $fh;
	open $fh, $file or die "failed to read $file: $!";
	my $data;
	read $fh, $data, -s $fh;
	close $fh;
	
	# modify
	for (my $i = 0; $i < @scan; $i += 2)
	{
		my $regex = $scan[$i];
		my $scanlist = $scan[$i + 1];

		eval "($scanlist) = \$data =~ m!$regex!si";
		die $@ if $@;

		addlog('DataScan', "OK $file [$regex] [$scanlist] [".eval("($scanlist)")."]")
	}
	
	return;
}

sub actionCallURL
{
	my (@urls) = @_;

	evalParam() for @urls;
	
	my $ua = LWP::UserAgent->new;
	$ua->timeout(10);
	$ua->env_proxy;
	$ua->cookie_jar( {} ); 

	for my $url (@urls)
	{
		my $response = $ua->get($url);
		
		addlog('CallURL', $response->is_success ? "OK $url\n" . $response->content : "Failure $url: " . $response->status_line);
	}
	
	return;
}

sub actionVersionRead
{
	my ($var, $vfile, $default) = @_;

	evalParam() for $vfile;
	
	my $version;
	
	if (-e $vfile)
	{
		my $fh;
		open $fh, $vfile or die "Failed to read version file [$vfile]: $!";
		$version = <$fh>;
		close $fh;
		$version =~ s/[\s\t\r\n]+//g;
	}
	else
	{
		if ($default)
		{
			$version = $default;
		}
		else
		{
			die "Version file $vfile not found";
		}
	}
	
	# normalize version (e.g. "jin-1-2" => "jin-001-002", needed for string comparison)
	
	$version =~ s/\-(\d+)(?=\D|$)/'-' . sprintf("%03u", $1)/ge;

	# save version in the variable
	eval qq($var = "$version");
	die $@ if $@;

	addlog('VersionRead', "OK $vfile $var $version");
	
	return;
}

sub actionTstampFile
{
	my ($file) = @_;

	evalParam() for $file;

	unlink $file if -e $file;
	
	my $fh;
	open $fh, ">$file" or die "failed to create timestamp file $file: $!";
	print $fh strftime("%Y%m%d%H%M%S", localtime(time));
	close $fh;

	addlog('TstampFile', "OK $file");
	
	return;
}

sub actionMessage
{
	my (@msg) = @_;

	evalParam() for @msg;

	outputMessage(join('<br>', @msg));
	
	return;
}
	

#
# helpers

sub evalParam
{
	s/"/\\"/g;
	$_ = eval qq("$_");
	die $@ if $@;
}
	
sub splitSqlFile
{
	my $sql = shift;

	my @ret;

	$sql =~ s/[\n\r]+$//;
    
	my $sql_len = length($sql);
    my $char = '';
    my $string_start = '';
    my $in_string = 0;

    for (my $i = 0; $i < $sql_len; ++$i)
	{
        $char = substr($sql, $i, 1);

        # We are in a string, check for not escaped end of strings except for
        # backquotes that can't be escaped
        if ($in_string)
		{
            for (;;)
			{
				$i = index($sql, $string_start, $i);
                
				# No end of string found -> add the current substring to the
                # returned array
                if ($i == -1)
				{
					push @ret, $sql;
					return \@ret;
                }
				# Backquotes or no backslashes before quotes: it's indeed the
                # end of the string -> exit the loop
                elsif ($string_start eq '`' || substr($sql, $i-1, 1) ne '\\')
				{
                    $string_start = '';
                    $in_string = 0;
                    last;
                }
                # one or more Backslashes before the presumed end of string...
                else
				{
                    # ... first checks for escaped backslashes
                    my $j = 2;
                    my $escaped_backslash = 0;
                    
					while ($i - $j > 0 && substr($sql, $i - $j, 1) eq '\\')
					{
                        $escaped_backslash = ! $escaped_backslash;
                        $j++;
                    }
                    
					# ... if escaped backslashes: it's really the end of the
                    # string -> exit the loop
                    if ($escaped_backslash)
					{
                        $string_start = '';
                        $in_string = 0;
                        last;
                    }
                    # ... else loop
                    else
					{
                        $i++;
                    }
                }
            }
        }

        # lets skip comments (/*, -- and #)
        elsif (($char eq '-' && $sql_len > $i + 2 && substr($sql, $i + 1, 1) eq '-' && substr($sql, $i + 2, 1) <= ' ') || $char eq '#' || ($char eq '/' && $sql_len > $i + 1 && substr($sql, $i + 1, 1) eq '*'))
		{

			$i = index($sql, $char eq '/' ? '*/' : "\n", $i);
            # didn't we hit end of string?
            if (! $i)
			{
                last;
            }
            $i++ if $char eq '/';
        }

        # We are not in a string, first check for delimiter...
        elsif ($char eq ';')
		{
            # if delimiter found, add the parsed part to the returned array
            push @ret, substr($sql, 0, $i);
            $sql = substr($sql, $i + 1 < $sql_len ? $i + 1 : $sql_len);
            $sql_len = length($sql);
            if ($sql_len)
			{
                $i = -1;
            }
			else
			{
                # The submited statement(s) end(s) here
                return \@ret;
            }
        }

        # ... then check for start of a string,...
        elsif (($char eq '"') || ($char eq '\'') || ($char eq '`'))
		{
            $in_string = 1;
            $string_start = $char;
        }

    }

    # add any rest to the returned array
    if ($sql =~ /\S/)
	{
        push @ret, $sql;
    }

    return \@ret;
}

#
# main program

init();

if ($In{installer_profile})
{
	loadConfig($In{installer_profile});

	outputMessage('', 1)
		unless $In{do_it};

	# doing Actions

	for my $a (@Actions)
	{
		die "invalid action [$a->{name}]"
			unless defined &{"action$a->{name}"};
		&{"action$a->{name}"}(@{$a->{param}});
	}
}
else
{
	loadProfiles();
	outputMessage('', 2);
}

1;
